<?php 

/*
** Fishlib - a collection of utilities for db-driven applications in PHP
** Copyright (C) 2002  LTWD, LLC DBA The Madfish Group
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

class FishText
{

	function charset($charset='',$mimetype='')
	{
		if (empty($charset))
		{
			$charset = 'ISO-8859-1';
		}
		if (empty($mimetype))
		{
			$mimetype = 'text/html';
		}

		header("Content-Type: $mimetype; charset=$charset");
	}


	// string cleanup_text ([string value [, string preserve [, string allowed_tags]]])

	// This function uses the PHP function htmlentities() to convert
	// special HTML characters in the first argument (e.g., &,",',<, and >) 
	// to the equivalent HTML entities. If the optional second argument is empty,
	// any HTML tags in the first argument will be removed. The optional
	// third argument lets you specify specific tags to be spared from
	// this cleansing. The format for the argument is "<tag1><tag2>".

	function cleanup_text($value='', $preserve='', $allowed_tags='')
	{
		if (empty($preserve)) 
		{ 
			$value = strip_tags($value, $allowed_tags);
		}
		$value = htmlentities($value);
		return $value;
	}

	function reverse_cleanup_text($value='')
	{
		static $reverse_entities = NULL;
		if ($reverse_entities === NULL)
		{
			$reverse_entities = array_flip(
				get_html_translation_table(HTML_ENTITIES)
			);
		}
		return strtr($value,$reverse_entities);
	}

	// string make_page_title ([string title])

	// This function will clean up a string to make it suitable for use
	// as the value of an HTML <TITLE> tag, removing any HTML tags and
	// replacing any HTML entities with their literal character equivalents.

	function make_page_title($title='')
	{
		return FishText::reverse_cleanup_text(FishText::cleanup_text($title));
	}

	function get_host($default=FALSE)
	{
		if (!empty($_SERVER['HTTP_HOST']))
			$host = $_SERVER['HTTP_HOST'];
		elseif (!empty($_SERVER['SERVER_NAME']))
			$host = $_SERVER['SERVER_NAME'];
		elseif (!empty($_SERVER['HOST']))
			$host = $_SERVER['HOST'];
		else
			$host = $default;
		return $host;
	}

	function get_local_ref($ref='')
	{
		if (substr($ref,0,1) != '/')
		{
			$ref = dirname($_SERVER['PHP_SELF']).'/'.$ref;
		}
		return $ref;
	}

	function make_url($ref='', $scheme='http', $port=NULL)
	{
		$host = FishText::get_host();
		if ($port)
		{
			$host = substr($host, 0, strstr($host, ':')).':'.$port;
		}
		$url = $scheme.'://'.$host.FishText::get_local_ref($ref);
		return $url;
	}

	// string regular_url ([string ref])

	// This function will transform a local URL into an absolute URL pointing
	// to a normal server running on the same domain, as defined by the global
	// Apache variable HTTP_HOST. (Note: the server we are using runs on 
	// non-standard ports, thus the need to change the port numbers.)

	function regular_url($ref='',$port=NULL)
	{
		// leave this blank to use default port (usually 80)
		$http_port = '';
		if (empty($port))
		{
			$port = $http_port;
		}
		return FishText::make_url($ref, 'http', $port);
	}

	// string secure_url ([string ref])

	// This function will transform a local URL into an absolute URL pointing
	// to a secure server running on the same domain.

	function secure_url($ref='',$port=NULL)
	{
		// leave this blank to use default port (i.e. leave unspecified)
		$ssl_port = '';
		if (empty($port))
		{
			$port = $ssl_port;
		}
		return FishText::make_url($ref, 'https', $port);
	}

	function labelize($in='')
	{
		static $known_caps = ' URL DEA FBI XML ';
		$bits = preg_split(
			'/([\w]+[\']*[\w]+)/'
			, $in
			, -1
			, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY
		);
		$mixed = preg_match('/[a-z]/',$in) && preg_match('/[A-Z]/',$in);
		$out = '';
		foreach ($bits as $b)
		{
			if (preg_match('/[a-z]/i',$b))
			{
				if (stristr($known_caps," $b ") !== FALSE)
				{
					$b = strtoupper($b);
				}
				elseif ($mixed && preg_match('/^[A-Z][A-Z]/',$b))
				{
	//print "<li>b is mixed case already, ($b), leave it alone.\n";
				}
				elseif (preg_match('/[aeiouy]/i',$b))
				{
					$b = ucwords(strtolower($b));
				}
				elseif (!$mixed)
				{
					$b = strtoupper($b);
				}
			}
			$out .= $b;
		}
		return $out;
	}

	function semivalidate_email($email='')
	{
		$email = (string)$email;
		return preg_match(
			'/^[a-z0-9\=\_\.\-]+@[a-z0-9\\._\-]+\.[a-z]{2,4}$/i'
			, $email
		);
	}

	function anchor_tag($url=NULL)
	{
		if (empty($url))
			return;
		$link_text = $url;
		if (strlen($url) > 30)
			$link_text = substr($link_text,0,30).'...';
		if (!preg_match('/^[a-z]+:\/\//', $url))
		{
			$url = "http://{$url}";
		}
		$tag = "<a href=\"$url\">$link_text</a>";
		return $tag;
	}

	function filter_blanks($list,$filter=FALSE)
	{
		return $filter ? array_diff($list, array('')) : $list;
	}

	function join($sep,$list,$filter=FALSE)
	{
		return implode($sep, FishText::filter_blanks($list, $filter));
	}

	function split($sep,$string,$filter=FALSE)
	{
		return FishText::filter_blanks(explode($sep, $string), $filter);
	}

}
?>
